# Non-State Actor (NSA) characteristics
# http://privatewww.essex.ac.uk/~ksg/eacd.html

## ------------------------------------------- ##
## NSA dataset (Cunningham / Gleditsch et.al.) ##
## ------------------------------------------- ##

nsa <- read.csv("Daten//Non-State Actor in Armed Conflict Dataset/nsa_v3.4_21November2013.tab", header = T, sep = "\t", as.is = T)

# Extracting all 'government-supporting' third party states in a conflict and giving them their own line
# Not helpful:
#   - 'rebstrength' is a dyadic value that is not correct for the dyads concerning the supporting governments.
#   - unclear weather the relative strength is calculated in taking third party help into account or not
# tmp <- nsa
# for (y in 1:nrow(nsa)) {
#   if (nsa[y,c("gsupname")]!="" & 
#       nsa[y,c("gsupname")]!="." & 
#       nsa[y,c("gsupname")]!=" " & 
#       !is.na(nsa[y,c("gsupname")]) & 
#       nsa[y,c("gtypesup")]!="non-military" & 
#       !is.na(nsa[y,c("gtypesup")])) {
#     names <- strsplit(nsa[y,c("gsupname")], ";")
#     #print(names)
#     for (i in 1:length(names[[1]])) {
#       new.row <- nsa[y,]
#       new.row$side_a <- as.character(names[[1]][i])
#       tmp <- rbind(tmp, new.row)
#     }
#   }
# }

nsa$actorid <- countrycode(nsa$side_a, "country.name", "cown")
nsa <- subset(nsa, select = c(ucdpid, actorid, side_a, startdate, enddate, strengthcent, rebestimate, rebstrength, mobcap, armsproc, fightcap, terrcont, effterrcont))

# extract start and end year
nsa$startyear <- year(nsa$startdate)
nsa$endyear <- year(nsa$enddate)

# keep only data from conflicts that are active between 1989 and 2009
nsa <- subset(nsa, subset = endyear >= 1989 & startyear <= 2009)

# recode categorical variables to numeric

## strengthcent = "the extent to which a central command exercises control over the constituent groups of an insurgent movement"
nsa$strengthcent[is.na(nsa$strengthcent)] <- 0
nsa$strengthcent[nsa$strengthcent == "low"] <- 1
nsa$strengthcent[nsa$strengthcent == "moderate"] <- 2
nsa$strengthcent[nsa$strengthcent == "high"] <- 3
nsa$strengthcent <- as.numeric(nsa$strengthcent)

## rebstrength = "strength of the rebel forces relative to the government forces"
nsa$rebstrength[nsa$rebstrength == "much weaker"] <- -2
nsa$rebstrength[nsa$rebstrength == "weaker"] <- -1
nsa$rebstrength[nsa$rebstrength == "parity"] <- 0
nsa$rebstrength[nsa$rebstrength == "stronger"] <- 1
nsa$rebstrength[nsa$rebstrength == "much stronger"] <- 2
nsa$rebstrength <- as.numeric(nsa$rebstrength)

## mobcap = "ability of the rebel group to mobilize personnel, rated relative to the government"
nsa$mobcap[nsa$mobcap == "no"] <- 0
nsa$mobcap[nsa$mobcap == "low"] <- 1
nsa$mobcap[nsa$mobcap == "moderate"] <- 2
nsa$mobcap[nsa$mobcap == "high"] <- 3
nsa$mobcap <- as.numeric(nsa$mobcap)

## armsproc = "the rebel group’s ability to procure arms, relative to the government"
nsa$armsproc[nsa$armsproc == "no"] <- 0
nsa$armsproc[nsa$armsproc == "low"] <- 1
nsa$armsproc[nsa$armsproc == "moderate"] <- 2
nsa$armsproc[nsa$armsproc == "high"] <- 3
nsa$armsproc <- as.numeric(nsa$armsproc)

## fightcap = "the rebel group’s fighting capacity, rated relative to the government"
nsa$fightcap[nsa$fightcap == "no"] <- 0
nsa$fightcap[nsa$fightcap == "low"] <- 1
nsa$fightcap[nsa$fightcap == "moderate"] <- 2
nsa$fightcap[nsa$fightcap == "high"] <- 3
nsa$fightcap <- as.numeric(nsa$fightcap)

## terrcont = "indicates whether the rebel group controls territory"
nsa$terrcont[nsa$terrcont == "no"] <- 0
nsa$terrcont[nsa$terrcont == "yes"] <- 1
nsa$terrcont <- as.numeric(nsa$terrcont)

## effterrcont = "indicates the degree of effective control the rebel group exercises over the territory"
nsa$effterrcont[is.na(nsa$effterrcont)] <- 0
nsa$effterrcont[nsa$effterrcont == "low"] <- 1
nsa$effterrcont[nsa$effterrcont == "moderate"] <- 2
nsa$effterrcont[nsa$effterrcont == "high"] <- 3
nsa$effterrcont <- as.numeric(nsa$effterrcont)


# collapse conflicts where several non-state actors are active (against one state actor)
nsa <- ddply(nsa, c(conflictid="ucdpid", "actorid", "side_a"), function(x) c(nsa.strengthcent.mean = round(mean(x$strengthcent, na.rm = T), digits = 0), # mean central command strength of insurgent groups
                                                          nsa.rebestimate.sum = sum(x$rebestimate, na.rm = T), # total estimated size of rebel armed forces
                                                          nsa.rebstrength.mean = round(mean(x$rebstrength, na.rm = T), digits = 0), # mean strength of rebel forces
                                                          nsa.mobcap.mean = round(mean(x$mobcap, na.rm = T), digits = 0), # mean capability of rebel forces to mobilize personnel
                                                          nsa.armsproc.mean = round(mean(x$armsproc, na.rm = T), digits = 0), # mean capability to procure arms
                                                          nsa.fightcap.mean = round(mean(x$fightcap, na.rm = T), digits = 0), # mean fight capacity of rebel groups
                                                          nsa.terrcont.max = max(x$terrcont, na.rm = T), # any territory controled by rebel groups ?
                                                          nsa.effterrcont.mean = round(mean(x$effterrcont, na.rm = T), digits = 0))) # mean degree of effective control the rebel group exercises over the territory

nsa[is.na(nsa)] <- NA
nsa$nsa.terrcont[is.infinite(nsa$nsa.terrcont)] <- NA

# Join with Master data set
Master <- join(Master, nsa, type="left", by=c("conflictid", "actorid"), match="all")

## CLEAN WORKSPACE
rm(nsa)